const WebSocket = require('ws');
const SocksProxyAgent = require('socks-proxy-agent');
const proxy = process.env.agent;
const agent = proxy ? new SocksProxyAgent(proxy) : null;
const request = require('request');
const host = 'https://api.kucoin.com';
const CryptoJS = require('crypto-js');
const constants = require('./constants');
const {IPs, mergeDepthAsk, mergeDepthBids, mergeDepth} = require('./util');
let totalOrderbook = {};
let cachedUpdate = {};
const fetchingMap = {};


let getUpdatedData = function (currentData, response) {
  const updateData = [];
  let newCount = 0;
  let has = false;
  for (const a of currentData) {
    if (response.data.price === a[0]) {
      has = true;
      newCount = a[1];
      if (response.data.action === 'ADD') {
        newCount += response.data.count;
      }
      else if (response.data.action === 'CANCEL') {
        newCount -= response.data.count;
        if (newCount < 0) {
          newCount = 0;
        }
      }
      break;
    }
  }
  if (!has) {
    if (response.data.action !== 'CANCEL')
      updateData.push([response.data.price, response.data.count]);
  } else {
    // updateData = [[response.data.price, newCount]];
    updateData.push([response.data.price, parseFloat(newCount).toFixed(8)]);
  }
  return updateData;
};

class biboxApi {
  constructor() {
    this.apiKey = '5ba054d2e0abb830dee81e1d';
    this.apiSecret = 'baa55ba9-7290-431e-9fd4-42aa4598887f';
    this.allowRequest = true;
    this.index = 0;
  }
  _selectIp(){
    const ip = IPs[this.index % IPs.length];
    this.index++;
    return ip;
  }

  subscribeSymbols(symbols, depth, callback) {
    const ip = this._selectIp();
    this._publicRequest('/api/v1/bullet-public', {}, (error, result) => {
      if (error) {
        console.log("fetch ws token error");
        this.subscribeSymbols(symbols, depth, callback);
      } else {
        this._openWs(result, symbols, depth, ip, callback);
      }
    }, ip, 'POST');
  }

  _openWs(result, symbols, depth, ipAddress, callback) {
    const token = result.data.token;
    const serverInfo = result.data.instanceServers[0];
    const pingInterval = serverInfo.pingInterval;
    const wsUrl = serverInfo.endpoint;
    const reqURL = `${wsUrl}?token=${token}&connectId=${Date.now()}`;
    const wss = new WebSocket(reqURL, {agent, localAddress: ipAddress});
    wss.on('open', () => {
      console.log("websocket on open");
    });
    wss.on('message', (data) => {
      const response = JSON.parse(data);
      if (response.type === 'welcome') {
        const id = response.id;
        setInterval(() => {
          if (wss.readyState === constants.SocketReadyStateOpen) {
            wss.send(JSON.stringify({id, type: 'ping'}))
          }
        }, pingInterval);
        const topics = symbols.join(",");
        wss.send(JSON.stringify({id,type:"subscribe",topic:`/market/level2:${topics}`,response:true}));
      } else if (response.topic && response.topic.startsWith('/market/level2')) {
        const symbol = response.data.symbol;
        const snapshot = totalOrderbook[symbol];
        if(!snapshot ){
          if(!fetchingMap[symbol]){
            fetchingMap[symbol] = true;
            setTimeout(()=>{
              this.getOrderbook(symbol,100,ipAddress,this._handleOrderbookSnapshot.bind(this,symbol));
            },100);
          }
          let array = cachedUpdate[symbol];
          if(!array){
            array = [];
            cachedUpdate[symbol] = array;
          }
          array.push(response.data);
        }else{
          const updatedAsks = response.data.changes.asks;
          const updatedBids = response.data.changes.bids;
          const mergedAsks = mergeDepth(snapshot.asks,updatedAsks,true);
          const mergedBids = mergeDepth(snapshot.bids, updatedBids, false);
          // if(symbol === 'BCHSV-BTC'){
          //   console.log(updatedBids.toString())
          //   console.log(snapshot.bids.toString());
          //   console.log(mergedBids.toString());
          //   console.log("====================")
          // }
          snapshot.asks = mergedAsks.slice(0,50);
          snapshot.bids = mergedBids.slice(0,50);
          callback(null,{symbol,asks:mergedAsks, bids:mergedBids, timestamp:Date.now()});
        }
      }

    });
    wss.on('error', (error) => {
      console.log("websocket error:");
      console.log(error);
    })
    wss.on('close', () => {
      console.log("websocket closed");
      setTimeout(() => {
        this._openWs(result, symbols, depth, ipAddress, callback);
      }, 2000);
    })
  }
  _handleOrderbookSnapshot(symbol,error,data){
    fetchingMap[symbol] = false;
    if(error){
      console.error("get snapshot error:"+symbol);
      console.error(error);
      return;
    }
    totalOrderbook[symbol] = data.data;
    const lupdates = cachedUpdate[symbol];
    for(const update of lupdates){
      const updatedAsks = update.changes.asks;
      const updatedBids = update.changes.bids;
      const filteredAsks = updatedAsks.filter((item)=>{
        return item[2] > data.data.sequence;
      });
      const filteredBids = updatedBids.filter((item)=>{
        return item[2] > data.data.sequence;
      })
      const mergedAsks = mergeDepth(totalOrderbook[symbol].asks,filteredAsks,true);
      const mergedBids = mergeDepth(totalOrderbook[symbol].bids, filteredBids, false);
      totalOrderbook[symbol].asks = mergedAsks;
      totalOrderbook[symbol].bids = mergedBids;
    }

  }

  // _subscribeSymbol(wss, id, symbol, depth) {
  //   const ipAddress = IPs[Math.round(Math.random() * (IPs.length - 1))];
  //   this.getOrderbook(symbol, depth, ipAddress, (error, result) => {
  //     if (error) {
  //       console.error("get symbol by rest error:");
  //       console.error(error);
  //       setTimeout(() => {
  //         this._subscribeSymbol(wss, id, symbol, depth);
  //       }, 2000);
  //       return;
  //     }
  //     const data = result.data;
  //     const oldData = totalOrderbook[symbol];
  //     if (!oldData || data.timestamp > oldData.timestamp) {
  //       totalOrderbook[symbol] = {
  //         asks: data.SELL.map((item) => [item[0], item[1]]),
  //         bids: data.BUY.map((item) => [item[0], item[1]]),
  //         timestamp: data.timestamp
  //       };
  //     }
  //     wss.send(JSON.stringify({id, type: 'subscribe', 'topic': `/trade/${symbol}_TRADE`}))
  //   });
  // }

  transform(obj) {
    var str = [];
    for (var p in obj)
      str.push(encodeURIComponent(p) + "=" + encodeURIComponent(obj[p]));
    return str.join("&");
  }

  objKeySort(obj) {//排序的函数
    var newkey = Object.keys(obj).sort();
    //先用Object内置类的keys方法获取要排序对象的属性名，再利用Array原型上的sort方法对获取的属性名进行排序，newkey是一个数组
    var newObj = {};//创建一个新的对象，用于存放排好序的键值对
    for (var i = 0; i < newkey.length; i++) {//遍历newkey数组
      newObj[newkey[i]] = obj[newkey[i]];//向新创建的对象中按照排好的顺序依次增加键值对
    }
    return newObj;//返回排好序的新对象
  }

  _publicRequest(path, params, callback, bindIP, method = 'GET') {
    if (!this.allowRequest) {
      callback({code: "-2", message: "出现超频情况，暂停提交请求"});
      return;
    }
    let url = host + path;
    if (params && method === 'GET') {
      url += "?";
      const keys = Object.keys(params);
      for (let i = 0; i < keys.length; i++) {
        url += `${keys[i]}=${params[keys[i]]}`;
        if (i !== keys.length - 1) {
          url += "&";
        }
      }
    }
    const options = {
      url,
      method: method,
      timeout: 8000,
      forever: true,
      localAddress: bindIP,

    };
    request(options, (error, response, body) => {
      if (error) {
        callback(error);
      } else {
        try {
          const result = JSON.parse(body);
          if (result.code === '200000') {
            callback(null, result);
          } else {
            callback(result, null);
          }
        } catch (e) {
          // console.error(e);
          // console.error(body);
          if (response.statusCode == 200) {
            console.log(body);
            console.log(e);
          }
          callback({statusCode: response.statusCode}, null);
        }
      }
    });
  }

  _request(method, path, params, callback) {
    if (!this.allowRequest) {
      callback({code: "-2", message: "出现超频情况，暂停提交请求"});
      return;
    }
    let url = host + path;
    const nonce = new Date().getTime();
    const strParams = this.transform(this.objKeySort(params));
    const strForSign = path + '/' + nonce + '/' + strParams;
    const strForSign2 = CryptoJS.enc.Base64.stringify(CryptoJS.enc.Utf8.parse(strForSign))
    const sign = CryptoJS.HmacSHA256(strForSign2, this.apiSecret).toString(CryptoJS.enc.Hex);
    const headers = {
      "Accept-Language": "zh_CN",
      'KC-API-KEY': this.apiKey,
      'KC-API-NONCE': nonce,
      'KC-API-SIGNATURE': sign
    }
    const requestParams = {};
    let form = params
    if (method === 'GET') {
      url += '?' + this.transform(params);
      form = {}
    }
    requestParams["url"] = url;
    requestParams["method"] = method;
    requestParams["headers"] = headers;
    requestParams["form"] = form;
    requestParams["timeout"] = 10000;
    requestParams["forever"] = true;
    request(requestParams, (error, response, body) => {
      if (error) {
        callback(error);
      } else {
        try {
          const result = JSON.parse(body);
          if (result && result.success && result.code === 'OK') {
            callback(null, result);
          } else {
            callback(result, null);
          }
        } catch (e) {
          console.error("parse body时出错");
          console.error("status code:" + response.statusCode);
          callback({statusCode: response.statusCode}, null)
          // if (response.statusCode === 429) {
          // callback({statusCode: response.statusCode})
          // } else {
          //   callback(e, null);
          // }
        }
      }
    });
  }


  fetchSymbols(callback) {
    this._publicRequest('/api/v1/symbols', {}, callback);
  }

  order(price, amount, symbol, side, callback) {
    //账户类型，0-普通账户，1-信用账户
    //交易类型，1-市价单，2-限价单
    //交易方向，1-买，2-卖
    const params = {
      symbol: symbol,
      type: side === constants.OrderSideBuy ? 'BUY' : 'SELL',
      pay_bix: 1,
      price,
      amount,
    };

    this._request("POST", "/v1/order", params, callback)
  }

  balance(limit, page, callback) {
    this._request("GET", "/v1/account/balances", {limit: limit || 20, page: page || 1}, callback);
  }

  coins_info(callback) {
    this._publicRequest("/v1/market/open/coins", {}, callback);
  }

  getTrades(symbol, callback) {
    callback(null, null)
  }

  searchOrder(orderId, symbol, side, callback) {
    const params = {
      symbol: symbol,
      type: side === constants.OrderSideBuy ? 'BUY' : 'SELL',
      orderOid: orderId
    }
    this._request("GET", "/v1/order/detail", params, callback);
  }

  cancelOrder(orderId, symbol, side, callback) {
    const params = {
      symbol: symbol,
      orderOid: orderId,
      type: side === constants.OrderSideBuy ? 'BUY' : 'SELL',
    }
    this._request("POST", "/v1/cancel-order", params, callback);
  }

  fetchHistorOrders(page, size, symbol, side, callback) {
    const params = {
      pair: symbol,
      account_type: 0,
      page,
      size,
      order_side: side === constants.OrderSideBuy ? 1 : 2,
      hide_cancel: 0
    };
    this._request("POST", "/v1/orderpending", "orderpending/pendingHistoryList", params, callback);
  }

  getOrderbook(symbol, depth, bindIP, callback) {
    this._publicRequest("/api/v1/market/orderbook/level2_100", {"symbol": symbol}, callback, bindIP);
  }
}

module.exports = biboxApi;