const BaseCollector = require('./baseCollector');
const baseCurrencies = ["BIX","ETH","BTC","USDT"];
const restCurrencies = ["PRA","AAC","AT","RED","BU","QTUM","EOSDAC","HPB","ORME","TRX","TTC","BIX","BTC","ETH"];
const machine = process.env['MACHINE'];
const biboxApi = require('./api');
const zlib = require('zlib');
const sendEmail = require('./mailer');
let bixNumMonitor = 0;
const Strategy3MaxAmountMap = {
  USDT: 1000,
  BTC: 0.15,
  ETH: 1.5,
  BIX:600
}


class BiboxCollector extends BaseCollector{
  constructor(wantedSymbols){
    super("BIBOX",baseCurrencies,machine,wantedSymbols);
    this.api = new biboxApi('','');
    setInterval(()=>{
      if(bixNumMonitor >=20){
        sendEmail("Bibox 平台币数量已少于80","平台币数量不够啦，先手动处理一下啦");
        bixNumMonitor = 0;
      }
    },1000 * 60 * 20);
  }
  _convertSymbolName(symbol){
    return symbol.replace("_","");
  }
  _fetchSymbols(callback){
    this.api.fetchSymbols((error,data)=>{
      if(error){
        console.error("fetch symbol error");
        console.error(error);
        this._fetchSymbols(callback);
      }else{
        const symbolDetails = data.result;
        const symbolMap = {};
        for(let detail of symbolDetails){
          symbolMap[detail.pair] = detail;
        }
        callback(null,symbolMap);
      }
    })
  }

  _runMonitorInterval(){
    return 4000;
  }

  _subscribeSymbols(symbols,callback,subscribeDepth){
    // this._fetchDepthByRest(symbols,subscribeDepth,callback);
    function subscribeCallback(error,data){
      if(error){
        console.error("subscribe error");
        console.error(error);
      }else if(data.channel.endsWith("depth")){
        const symbol = data.channel.replace("bibox_sub_spot_","").replace("_depth","").replace("_","");
        const decodeStr = Buffer.from(data.data,'base64');
        const unzipData = zlib.unzipSync(decodeStr);
        const result = JSON.parse(unzipData.toString());
        // const symbol = result.pair.replace("_","");
        const timeStamp = result.update_time;
        const asks = result.asks.slice(0,subscribeDepth).map((item)=>[item.price,item.volume]);
        const bids = result.bids.slice(0,subscribeDepth).map((item)=>[item.price,item.volume]);
        callback(asks, bids, symbol, timeStamp);
        // console.log("深度延时："+(Date.now()-timeStamp));
      }else{
        const ticker = data.data;
        const symbol = ticker.pair.replace("_","");
        const bids = [[ticker.buy,ticker.buy_amount]];
        const asks = [[ticker.sell,ticker.sell_amount]];
        callback(asks,bids,symbol,ticker.timestamp);
        console.log(symbol + "ticker 延时："+(Date.now()-ticker.timestamp));
      }
    }
    for(let i=0;i<7;i++){
      setTimeout(()=>{
        this.api.subscribeSymbolsAndTicker(symbols,subscribeCallback);
      },i*30000);
    }
  }
  _fetchDepthByRest(symbols,depth,callback){
    const restSymbols = [];
    for(let symbol of symbols){
      const midCurrency = symbol.split("_")[0];
      if(restCurrencies.includes(midCurrency)){
        restSymbols.push(symbol);
      }
    }
    const perInterval = 500;
    const totalInterval = perInterval * restSymbols.length+50;
    setInterval(()=>{
      const sortedSymbols = restSymbols.sort(()=>{
        return Math.random()>0.5
      });
      for(let i=0;i<sortedSymbols.length;i++){
        const symbol = sortedSymbols[i];
        setTimeout(()=>{
          this.api.getOrderbook(symbol,depth,(error,result)=>{
            if(error){
              console.error("get depth by rest error:");
              console.error(error);
              return;
            }
            const data = result.result;
            const timeStamp = data.update_time;
            callback(data.asks.map((item)=>[item.price,item.volume]),data.bids.map((item)=>[item.price,item.volume]),symbol,timeStamp);
          })
        },i*perInterval);
      }
    },totalInterval);
  }
  _runMonitor(callback){
    this.api.balance((error,result)=>{
      if(error){
        console.error("get balance error");
        console.error(error);
        callback(null);
      }else{
        const balanceList = result.result.assets_list;
        const balanceMap = {};
        for(let detail of balanceList){
          if(detail.balance >0 || detail.freeze >0 || balanceMap[detail.coin_symbol]){
            balanceMap[detail.coin_symbol] = {available:detail.balance,onOrder:detail.freeze};
          }
        }
        callback(balanceMap);
        if(this.getCurrencyBalance("BIX",true)<80){
          bixNumMonitor ++ ;
        }else {
          bixNumMonitor >0 ?bixNumMonitor-- : bixNumMonitor=0;
        }
      }
    })
  }

  getSymbol(fromCurrency,toCurrency){
    let symbolDetail = this._getSymbolDetail(fromCurrency,toCurrency);
    return symbolDetail.pair;
  }
  getFeeRate(fromCurrency,toCurrency){
    return 0.0005;
  }
  processAmount(fromCurrency,toCurrency,amount){
    //有可能会由于数值过小，传进来的amount被表示为科学计数法法
    const amountStr = parseFloat(amount).toFixed(10);
    const nums = amountStr.split('.');
    if(nums.length ===2 && nums[1].length>4){
      return nums[0]+"."+nums[1].slice(0,4);
    }
    return amount;
  }
  processPrice(fromCurrency,toCurrency,price){
    // const amountStr = price+'';
    // const nums = amountStr.split('.');
    // if(nums.length ===2 && nums[1].length>4){
    //   return nums[0]+"."+nums[1].slice(0,4);
    // }
    return price;
  }
  getCurrencyMaxReturnAmount(currency){
    const balance = this.getCurrencyBalance(currency);
    return Math.min(Strategy3MaxAmountMap[currency],balance);
  }
  getDepthPrintStr(fromCurrency,toCurrency,depth=1){
    return super.getDepthPrintStr(fromCurrency,toCurrency,3);
  }
  _allowPublish(asks,bids,symbol){
    const balance = this.getCurrencyBalance("BIX");
    if(balance < 80){
      console.error("目前BIX数量少于80，暂停交易");
      return false;
    }
    return super._allowPublish(asks,bids,symbol);
  }

}

module.exports = BiboxCollector;

