let symbolMap = {};
const redis = require("redis");
const redisClient = redis.createClient();
const lastDataMap = {};
const lastLastDataMap = {};
const constants =require('./constants');

class BaseCollector {
  constructor(platformName, baseCurrencies,machine,str2Symbols) {
    this.onSymbolUpdate = null;
    this.allowTrade = true;
    this.baseCurrencies = baseCurrencies;
    this.platformName = platformName;
    this.currencySymbolMap = {};
    this.machine = machine;
    this.balanceMap = {};
    this.str2Symbols = str2Symbols;
    this.balanceException = false;
  }

  /**
   *
   * @param asks [[price,amount],[price,amount]] 升序排序
   * @param bids [[price,amount],[price,amount]] 降序排序
   * @param timeStamp 从服务器获取的数据的时间戳
   * @private
   */
  _publishDataForStrategy3(asks, bids, symbol, timeStamp,version) {
    if(!asks.length && !bids.length){
      return;
    }
    const realSymbol = this._convertSymbolName(symbol);
    const lastEventTime = this.getSymbolEventTime(realSymbol);
    const lastVersion = this._getLastDataVersion(realSymbol);
    if(lastVersion >0 && version){
      if(lastVersion >= version){
        return;
      }
    } else if(lastEventTime >= timeStamp){
      return;
    }

    const data = {
      bestBidQty: bids[0][1],
      bestAskQty: asks[0][1],
      eventTime: timeStamp,
      bestBid: bids[0][0],
      bestAsk: asks[0][0],
      symbol: realSymbol,
      asks: asks,
      bids: bids,
      version:version,
    }
    if(parseFloat(data.bestBid) > parseFloat(data.bestAsk)){
      // console.error("同一交易对，卖价低于买价，可能数据有误:"+realSymbol);
      return;
    }
    const key = `${realSymbol}@${this.platformName.toLowerCase()}0`;
    redisClient.set(key, JSON.stringify(data), 'EX', 120);
    const lastData = lastDataMap[realSymbol];
    let dataChanged = true;
    if (lastData) {
      dataChanged = lastData.bestBidQty !== data.bestBidQty
          || lastData.bestAskQty !== data.bestAskQty || lastData.bestBid !== data.bestBid || lastData.bestAsk !== data.bestAsk
      ||lastData.asks.toString() !== data.asks.toString() || lastData.bids.toString() !== data.bids.toString();
    }
    if (dataChanged) {
      lastDataMap[realSymbol] = data;
      lastLastDataMap[realSymbol] = lastData;
      const publistKey = `${this.platformName.toUpperCase()}0`;
      redisClient.publish(publistKey, JSON.stringify({symbol: realSymbol, time: data.eventTime}));

      if (this.onSymbolUpdate && this.allowTrade && this._allowPublish(asks,bids,symbol) ) {
        this.onSymbolUpdate(realSymbol, "", Date.now() - data.eventTime);
      }
    }
  }

  _getSymbolDetail(fromCurrency,toCurrency){
    let symbol = fromCurrency+toCurrency;
    let detail = symbolMap[symbol];
    if(detail){
      return detail;
    }
    symbol = toCurrency+fromCurrency;
    return symbolMap[symbol];
  }

  getBaseCurrency(symbol) {
    for (const baseCurrency of this.baseCurrencies) {
      if (symbol.endsWith(baseCurrency.toUpperCase())) {
        return baseCurrency.toUpperCase();
      }
    }
  }

  getMidCurrency(symbol) {
    for (const baseCurrency of this.baseCurrencies) {
      if (symbol.endsWith(baseCurrency.toUpperCase())) {
        return symbol.replace(new RegExp(baseCurrency.toUpperCase() + '$'), '');
      }

    }
  }
  getSymbols(currency) {
    if (!currency || !this.currencySymbolMap[currency]) {
      return [];
    }
    return this.currencySymbolMap[currency].sort(() => {
      return Math.random() > 0.5;
    })
  }

  getDepth(fromCurrency, toCurrency, depth = 1) {
    let symbol = fromCurrency + toCurrency;
    let data = lastDataMap[symbol];
    if (data) {
      const bids = data.bids;
      return bids.slice(0, depth);
    }
    else {
      symbol = toCurrency + fromCurrency;
      data = lastDataMap[symbol];
      if (!data) {
        return 0;
      }
      const asks = data.asks;
      return asks.slice(0, depth);
    }
  }

  getLastDepth(fromCurrency, toCurrency, depth=1) {
    let symbol = fromCurrency + toCurrency;
    let data = lastLastDataMap[symbol];
    if (data) {
      const bids = data.bids;
      return bids.slice(0, depth);
    }
    else {
      symbol = toCurrency + fromCurrency;
      data = lastLastDataMap[symbol];
      if (!data) {
        return 0;
      }
      const asks = data.asks;
      return asks.slice(0, depth);
    }
  }

  getSymbolEventTime(symbol) {
    const keySymbol = this._convertSymbolName(symbol);
    let lastData = lastDataMap[keySymbol];
    if(lastData){
      return lastData['eventTime'] || 0;
    }
    return 0;
  }
  _getLastDataVersion(symbol){
    const keySymbol = this._convertSymbolName(symbol);
    let lastData = lastDataMap[keySymbol];
    if(lastData){
      return lastData['version'] || -1;
    }
    return -1;
  }

  isBaseCurrency(currency) {
    return this.baseCurrencies.includes(currency);
  }

  getAmount(fromCurrency, toCurrency, depth = 1) {
    let symbol = fromCurrency + toCurrency;
    let data = lastDataMap[symbol];
    if (data) {
      const bids = data.bids;
      return bids.length>=depth ?[bids[depth - 1][1], fromCurrency]:-1;
    }
    else {
      symbol = toCurrency + fromCurrency;
      data = lastDataMap[symbol];
      if (!data) {
        return 0;
      }
      const asks = data.asks;
      return asks.length>=depth ?[asks[depth - 1][1], toCurrency]:-1;
    }
  }

  getPrice(fromCurrency, toCurrency, depth = 1, isRaw = false) {
    let symbol = fromCurrency + toCurrency;
    let data = lastDataMap[symbol];
    if (data) {
      const bids = data.bids;
      return bids.length>=depth?bids[depth - 1][0]:-1;
    }
    else {
      symbol = toCurrency + fromCurrency;
      data = lastDataMap[symbol];
      if (!data) {
        return -1;
      }
      const asks = data.asks;
      if (isRaw)
        return asks.length>=depth?asks[depth - 1][0]:-1;
      if (asks.length >= depth && asks[depth - 1][0])
        return 1.0 / asks[depth - 1][0];
      else
        return -1;
    }

  }

  _handleFetchSymbolResult(error,data){
    if(error){
      console.error("获取交易对出错");
      console.error(error);
      this._fetchSymbols(this._handleFetchSymbolResult.bind(this));
      return;
    }
    symbolMap={};
    const symbolNames = [];
    for(let key in data){
      if(!data.hasOwnProperty(key)){
        continue;
      }
      symbolNames.push(key);
      const symbol = this._convertSymbolName(key);
      symbolMap[symbol]=data[key];
      const midCurrency = this.getMidCurrency(symbol);
      const symbolArray = this.currencySymbolMap[midCurrency];
      if (symbolArray) {
        symbolArray.push(symbol);
      } else {
        this.currencySymbolMap[midCurrency] = [symbol];
      }
    }
    this._subscribeSymbols(symbolNames,this._publishDataForStrategy3.bind(this),5);
  }

  runStrategy3(){
    console.log("启动成功");
    setInterval(()=>{
      redisClient.get(this.machine + "_STOP", (err, value) => {
        if (value) {
          this.allowTrade = false;
          console.log("滑点过多，停止交易");
        } else {
          this.allowTrade = true;
        }
      });
    },1000 *10);
    this._runMonitor((data)=>{
      if(data){
        this.balanceMap = {...data};
        redisClient.set(this.machine,JSON.stringify(this.balanceMap),"EX",30);
        this.balanceException = false;
      }else{
        this.balanceException = true;
      }
    });
    setInterval(()=>{
      this._runMonitor((data)=>{
        if(data){
          this.balanceMap = {...data};
          redisClient.set(this.machine,JSON.stringify(this.balanceMap),"EX",30);
          this.balanceException = false;
        }else {
          this.balanceException = true;
        }
      });
    },this._runMonitorInterval());

    this._fetchSymbols(this._handleFetchSymbolResult.bind(this));
  }

  runStrategy2(){
    // this.runWebserviceDepth();
    // this.runMonitorForever();
    this._subscribeSymbols(this.str2Symbols,(asks,bids,symbol,timeStamp)=>{
      const key = symbol + `DEPTH@${this.platformName.toLowerCase()}`;
      console.warn(key);
      redisClient.set(key, JSON.stringify({
        'symbol': symbol,
        'asks': asks,
        'bids': bids,
      }), "EX", 15);
      redisClient.publish(`${this.platformName.toUpperCase()}@2`, JSON.stringify({symbol: symbol, time: Date.now()}));
    },5);
    setInterval(() => {
      this._runMonitor((data)=>{
        if(!data){
          redisClient.hset("balance", this.platformName.toLowerCase(), JSON.stringify({}));
        }else{
          redisClient.hset("balance", this.platformName.toLowerCase(), JSON.stringify(data));
        }
      });
    }, 2000);
  }

  getCurrencyMaxReturnAmount(currency){
    throw "应该覆盖此方法，返回该币种的最大回归量，主要用于规避风险；只有基础币需要设定此值";
  }

  processAmount(fromCurrency, toCurrency, amount){
    throw "应覆盖此方法，按照平台规则对通量的精度进行处理";
  }

  processPrice(fromCurrency,toCurrency,amount){
    throw "应覆盖此方法，按照平台规则对价格的精度进行处理";
  }
  getFeeRate(fromCurrency,toCurrency){
    throw "应覆盖此方法，返回交易手续费";
  }

  /**
   *
   * @param callback function:(error,data),data应该是：{symbol:{}}
   */
  _fetchSymbols(callback){
    throw "应覆盖fetchSymbols(callback)，从平台接口获取所有交易对"
  }

  /**
   *
   * @param callback function(data),data格式：{currency:{available,onOrder}},currency要求大写
   */
  _runMonitor(callback){
    throw "覆盖方法_runMonitor(callback),获取底仓，并返回结果"
  }

  _runMonitorInterval(){
    return 10 *1000;
  }

/**
   * @param symbols 数组，需要订阅交易对
   * @param callback function:(asks, bids, symbol, timeStamp)
   */
  _subscribeSymbols(symbols, callback, subscribeDepth){
    throw "应覆盖方法subscribeSymbols(symbols,callback)，订阅交易对，并将数据回调";
  }

  /**
   *
   * @param symbol 此值的格式由子类在覆盖fetchSymbols方法时，调用callback所返回的数据的中的symbol决定
   */
  _convertSymbolName(symbol){
    throw "覆盖方法convertSymbolName(symbol),将symbol转换成如BTCUSDT的格式";
  }

  getSymbol(fromCurrency, toCurrency) {
    throw "覆盖方法getSymbol(fromCurrency, toCurrency),按照平台API格式拼接出symbol";
  }

  getTradeSide(fromCurrency,toCurrency){
    let symbol = fromCurrency+toCurrency;
    const detail = symbolMap[symbol];
    if(detail){
      return constants.OrderSideSell;
    }
    return constants.OrderSideBuy;
  }
  _getLastData(symbol){
    return lastDataMap[this._convertSymbolName(symbol)];
  }

  getDepthPrintStr(fromCurrency,toCurrency,depth=1){
    const depthArray = this.getDepth(fromCurrency,toCurrency,depth);
    let depthStr = "";
    for(let i =0;i<depthArray.length;i++){
      depthStr += depthArray[i].toString();
      if(i!== depthArray.length-1){
        depthStr +="<br/>";
      }
    }
    return depthStr;
  }

  getLastDepthPrintStr(fromCurrency,toCurrency,depth=1){
    const depthArray = this.getLastDepth(fromCurrency,toCurrency,depth);
    let depthStr = "";
    for(let i =0;i<depthArray.length;i++){
      depthStr += depthArray[i].toString();
      if(i!== depthArray.length-1){
        depthStr +="<br/>";
      }
    }
    return depthStr;
  }

  /**
   * 添加额外的限制所有交易的条件
   */
  _allowPublish(asks,bids,symbol){
    const midCurrency = this.getMidCurrency(symbol);
    const balance = this.balanceMap[midCurrency];
    if(!this.isBaseCurrency(midCurrency) && balance && balance.onOrder >0){
      return false;
    }
    // if(!asks.length >=5 || !bids.length >=5){
    //   console.error("深度不足，放弃此轮");
    //   return false;
    // }
    return !this.balanceException;
  }

  getCurrencyBalance(currency, includeOnOrder = false) {
    const detail = this.balanceMap[currency.toUpperCase()];
    if (detail) {
      return includeOnOrder ? parseFloat(detail.available) + parseFloat(detail.onOrder) : parseFloat(detail.available);
    }
    return 0;
  }



}

module.exports = BaseCollector;


